/******************************************************************************
* FILENAME:      LPC2000_CAN_SYS.C                                            *  
*                                                                             *
* VERSION:       V1.0                                                         *
*                                                                             *
* DESCRIPTION:   NXP LPC2300 CAN Controller Driver with SJA1000               *                     
*	        	 Demo by UART0 with KEIL MCB2300 Evaluation Board	          *
*	        	 Code runing indicate by LED4 with Timer1 interrupt	       	  *
*                UART0 tested with Receive "a" NOT LED1, Receive "b" NOT LED2 *
*                                                                             *
* TOOLS:         KEIL uVision3                                     			  *
*                KEIL MCB2300 Evaluation Board               				  *
*                                                                             *
* REVISION HISTORY                                                            *
* Version  Author           Date        Remarks								  *
* 1.0      Anthony Xie      10/10/2005  CAN Driver without Full CAN           *
* 2.0      Antonio Ake      06/22/2007	Full CAN Driver			   			  *
* 2.1      Antonio Ake      10/11/2007  Add target MCB2300 RAM				  *
******************************************************************************/

// -----------------------------------------------
// Project include files                                                     
// -----------------------------------------------
#include "LPC230x.h"                            /* LPC21xx definitions */
#include "type.h"	
#include "target.h"
#include "LPC2000_CAN.h"
#include "irq.h"

//-----------------------------------------------
//Constant Definitions
//-----------------------------------------------
#define	  LED4CON	(1<<3)		// P2.3 control LED4
#define   LED3CON	(1<<2)		// P2.2 control LED3
#define	  LED2CON	(1<<1)		// P2.1 control LED2
#define   LED1CON	(1<<0)		// P2.0 control LED1

#define IER_RBR		0x01
#define IER_THRE	0x02
#define IER_RLS		0x04

//---------------------------------------------
//Global Variable Definitions
//---------------------------------------------

//System Global Variable
UInt32 System_Counter,LED_Counter=0;
UInt8 LED_Flag=0;
UInt8 System_Status=0;

UInt32 CANStatus;
UInt32 CAN1RxCount = 0;
UInt32 CAN2RxCount = 0,CAN2_FULLCANRxCount=0;
UInt32 CAN1ErrCount = 0, CAN2ErrCount = 0;
typedef struct
{
unsigned int Dat1; // Bits 0..10: CAN Message ID
				   // Bits 13..15: CAN interface number (1..4)
				   // Bits 16..19: DLC - Data Length Counter
				   // Bits 24..25: Semaphore bits
unsigned int DatA; // CAN Message Data Bytes 0-3
unsigned int DatB; // CAN Message Data Bytes 4-7

} FULLCAN_MSG;

//---------------------------------------------
//* Functions
//---------------------------------------------
void init(void);
void delay(UInt32);
void __irq IRQ_Timer1(	void);
void uart0_config(UInt32,UART0MODE);
void __irq IRQ_UART0(void);
void __irq IRQ_CAN(void);

void CAN_Rx2( void );
void CAN_FULLCAN_Rx2( void );
unsigned char msg_count = 0;



//***************************************************************************
//* main()
//* Input(s): none.
//* Returns : none.
//* Description : main loop
//***************************************************************************
int  main(void)
{ 
  UInt32 i=0;
  plpc2000CANdriver_TXObj_t pCAN_Send_Data;
  plpc2000CANdriver_RXObj_t pCAN_Rcv_Data;
    
  delay(20);
  init();  
  
  
  Print_Chars_to_Screen( Demo_Start );

  for(;;)	
  {        

    if( LED_Flag == 0 )
       FIO2SET = LED4CON;	//Control LED4
    else 
       FIO2CLR = LED4CON;

	    switch( System_Status )
	  {
	    case 0:	    // wait input (1,2 choice) status
		  break;
		case 1:									   // "1" be pressed
		  lpc2000CANdriver_SetACFMode( LPC2000_ACC_BYPASS );
		  Print_Chars_to_Screen( Demo_Sent_Data );
		  Print_Chars_to_Screen( Data_Sent_Start_or_Not ); 
		  System_Status = 4;
		  break;
		case 2:									   // "2" be pressed
          lpc2000CANdriver_SetACFMode( LPC2000_ACC_ON );
		  Print_Chars_to_Screen( Demo_Sent_Data );
		  Print_Chars_to_Screen( Acceptance_Filter_Tables );
		  Print_Chars_to_Screen( Data_Sent_Start_or_Not ); 
		  System_Status = 4;
		  break;
		case 3:									   // "3" be pressed
          lpc2000CANdriver_SetACFMode( LPC2000_ACCF_FULLCAN );
		  FCANIE = 0x1; 	       //Enable FullCAN Interrupt
		  Print_Chars_to_Screen( Demo_Sent_Data );
		  Print_Chars_to_Screen( FULLCAN_Acceptance_Filter_Tables );
		  Print_Chars_to_Screen( Data_Sent_Start_or_Not ); 
		  System_Status = 4;
		  break;
		case 4:	   // wait input (y,n choice) status
		  break;

		case 5:	   // "y or Y" be pressed, 
		  pCAN_Send_Data = CAN_Send_Data;
          	  for( i=0;i<11;i++)
	            {
                      Return_Value = lpc2000CANdriver_CertainTxBufTransmitMessage(LPC2000_CANCHANNEL_1,pCAN_Send_Data,LPC2000_TXB_1);
		      		  pCAN_Send_Data++;
		    }

		  System_Status = 0;
		  break;				
		  				
		case 7:	 // "N or n " be pressed
		  Print_Chars_to_Screen( Demo_Sent_Choice );
		  System_Status = 0;
		  break;
		default:
		  break;
	  }

    if( CAN_Rcv_Status == 1 ) //communication successed
	  {
          Print_Chars_to_Screen( Rcv_Data_OK );          
		  pCAN_Rcv_Data = CAN_Rcv_Data;	  // print to uart0 all data in Rcv_Buffer
		  for( i=0;i<CAN_Rcv_Data_Counter;i++)				  
		    {
			  
			  Rcv_Data_Output_to_Screen( pCAN_Rcv_Data );
			  pCAN_Rcv_Data++; 
			}
		  Print_Chars_to_Screen( Label_Explanation );	
		  CAN_Rcv_Data_Counter = 0;     // reset Rcv_Buffer, point to head, means empty it.
		  Print_Chars_to_Screen( Demo_Sent_Choice );

		  CAN_Rcv_Status = 0;
	  }


    if( CAN2Rx_Return_Message_Flag[0] != 0 ) // if one CAN1Rx interrupt happened
	  {										 // print massage of communication status
		
		for( i=1;i<10;i++)
		{
		  if( CAN2Rx_Return_Message_Flag[i] != 0)
		    {
              switch(i)
			    {
				  case 1: 
				    Print_Chars_to_Screen( CAN2Rx_Return_Message_EI );
				    Print_Chars_to_Screen( CAN_ERROR_STOP_DEMO );      
				    break;
 				  case 2: 
				    Print_Chars_to_Screen( CAN2Rx_Return_Message_DOI );
				    Print_Chars_to_Screen( CAN_ERROR_STOP_DEMO );
					break;
				  case 3: 
				    Print_Chars_to_Screen( CAN2Rx_Return_Message_WUI );
				    Print_Chars_to_Screen( CAN_ERROR_STOP_DEMO );
					break;
				  case 4: 
				    Print_Chars_to_Screen( CAN2Rx_Return_Message_EPI );
				    Print_Chars_to_Screen( CAN_ERROR_STOP_DEMO );
					break;
				  case 5: 
				    Print_Chars_to_Screen( CAN2Rx_Return_Message_ALI );
				    Print_Chars_to_Screen( CAN_ERROR_STOP_DEMO );
					break;
				  case 6: 
				    Print_Chars_to_Screen( CAN2Rx_Return_Message_BEI );
				    Print_Chars_to_Screen( CAN_ERROR_STOP_DEMO );
					break;
				  default:
				    break;
				}
			  CAN2Rx_Return_Message_Flag[i]=0;			  
			}
		}

	    CAN2Rx_Return_Message_Flag[0] = 0;
	  }

    if( (Return_Value != 0) && (Return_Value != 5))	  // Report CAN error status
	  {
	    switch(Return_Value)
		  {
		    case 10:
			  Print_Chars_to_Screen( CAN_Error_10 );
			  Print_Chars_to_Screen( CAN_ERROR_STOP_DEMO );
			  break;
		    case 30:
			  Print_Chars_to_Screen( CAN_Error_30 );
			  Print_Chars_to_Screen( CAN_ERROR_STOP_DEMO );
			  break;
		    case 31:
			  Print_Chars_to_Screen( CAN_Error_31 );
			  Print_Chars_to_Screen( CAN_ERROR_STOP_DEMO );
			  break;
		    case 40:
			  Print_Chars_to_Screen( CAN_Error_40 );
			  Print_Chars_to_Screen( CAN_ERROR_STOP_DEMO );
			  break;
		    case 50:
			  Print_Chars_to_Screen( CAN_Error_50 );
			  Print_Chars_to_Screen( CAN_ERROR_STOP_DEMO );
			  break;
		    case 60:
			  Print_Chars_to_Screen( CAN_Error_60 );
			  Print_Chars_to_Screen( CAN_ERROR_STOP_DEMO );
			  break;
		    case 90:
			  Print_Chars_to_Screen( CAN_Error_90 );
			  Print_Chars_to_Screen( CAN_ERROR_STOP_DEMO );
			  break;
		    case 100:
			  Print_Chars_to_Screen( CAN_Error_100 );
			  Print_Chars_to_Screen( CAN_ERROR_STOP_DEMO );
			  break;
		    case 110:
			  Print_Chars_to_Screen( CAN_Error_110 );
			  Print_Chars_to_Screen( CAN_ERROR_STOP_DEMO );
			  break;
		    case 120:
			  Print_Chars_to_Screen( CAN_Error_120 );
			  Print_Chars_to_Screen( CAN_ERROR_STOP_DEMO );
			  break;
		    case 130:
			  Print_Chars_to_Screen( CAN_Error_130 );
			  Print_Chars_to_Screen( CAN_ERROR_STOP_DEMO );
			  break;
			default:
			  break;
		  }

		Return_Value = 0;
	  }
      
  }
   
  
}

                               
//***************************************************************************
//* init()
//* Input(s): none.
//* Returns : none.
//* Description : initialization each function
//***************************************************************************
void init(void)
{
  UART0MODE Uart0_Config;


//-----------------------------------------------
//Config GPIO
//-----------------------------------------------
  PINSEL0  = 0x00000000;		// All be GPIO, Note: 2 bits config 1 pin!
  PINSEL0 |= 0x00000A55;		// P0.0-CANRD1,P0.1-CANTD1,P0.2-UARTTXD0,
  				             	// P0.3-UARTRXD0,P0.4-CANRD2,P0.5-CANTD2


  FIO2DIR = 0x000000FF;          // P2.0-7 output, other input, 
                                 // 0000 0000 0000 0000 0000 0000 0000 0000
  FIO2CLR = 0x000000FF;          // Turn off LED1 - LED8


//-----------------------------------------------
//Config Timer1
//Fcclk = Fosc*4 = 12MHz*4 = 48MHz
//Fpclk = Fcclk/2 = 48MHz/2 = 24MHz
//-----------------------------------------------  
  T1PR = 0;				// Disable prescaler
  T1MCR = 0x03;			// Using T1MR0 match T1TC,Enable interrupt,reset T1TC
  T1MR0 = Fpclk/100;	// 10ms each Timer interrupt
  T1TCR = 0x02;			// Reset T1TC, by set TCR:2
  T1TCR = 0x01;			// Must CLR TCR:2, then can Start T1TC


//-----------------------------------------------
//Config UART0
//-----------------------------------------------  
  Uart0_Config.Datalength = 8;
  Uart0_Config.Stopbit = 1;
  Uart0_Config.Paritybit = 0;
  uart0_config(9600,Uart0_Config);
  
  

//-----------------------------------------------
//Config CAN1
//-----------------------------------------------  
  lpc2000CANdriver_CANInit( LPC2000_CANCHANNEL_1, LPC2000_CANDRIVER_CANBITRATE33K24MHZ );
  
  
  //CAN1IER = 0x1;  //Enable receive CAN1 RX interrupt

  


//-----------------------------------------------
//Config CAN2
//-----------------------------------------------  
  lpc2000CANdriver_CANInit( LPC2000_CANCHANNEL_2, LPC2000_CANDRIVER_CANBITRATE33K24MHZ );

  CAN2IER = 0x1;  //Enable receive CAN2 RX interrupt
  
  lpc2000CANdriver_LoadAcceptanceFilter();

//-----------------------------------------------
//Config VIC, 
//In <Starup.s> enable IRQ by CLR I bit in CPSR
//Vectored interrupt Timer1 by slot-0
//Vectored interrupt UART0 by slot-1
//Vectored interrupt CAN1Rx by slot-2
//-----------------------------------------------  
 
 install_irq(TIMER1_INT,(void *)IRQ_Timer1 , 1 );
 install_irq(UART0_INT, (void *)IRQ_UART0  , 2 );
 install_irq(CAN_INT,   (void *)IRQ_CAN    , 3 );

}

/****************************************************************************
* Name: Delay()
* Function: Delay 
* Input(s): delaytime		30 about 0.5s
* Returns : none
****************************************************************************/
void  delay(UInt32  delaytime)
{  UInt32  i;

   for(; delaytime>0; delaytime--) 
      for(i=0; i<50000; i++);
}

/****************************************************************************
* Name: uart0_config(UInt32,UART0MODE)
* Function: UART0 config 
* Input(s): Baudrate
*           U0_config
* Returns : none
****************************************************************************/
void uart0_config(UInt32 Baudrate,UART0MODE U0_config)
{
  UInt32 temp_uart0;
//-------------------------
// UART0 Baudrate config
//-------------------------
  U0LCR = 0x80;			// DLAB=1, Can use U0DLM, U0DLL 
  temp_uart0 = (Fpclk/16)/Baudrate;
  
  U0DLM = temp_uart0>>8;
  U0DLL = temp_uart0&0xFF;
  
//-------------------------
// UART0 mode config
//------------------------- 
  temp_uart0 = U0_config.Datalength - 5;				// Data Length prepare
  
  if( U0_config.Stopbit == 2 ) temp_uart0 |= 0x04;		// Data Stop bit prepare
  
  if( U0_config.Paritybit != 0 )						// Data Parity bit prepare
    {
      U0_config.Paritybit = U0_config.Paritybit - 1;
      temp_uart0 |= 0x08;
    }
  temp_uart0 |= U0_config.Paritybit<<4;
  
  
  U0LCR = temp_uart0; 	// Data Length, Stop bit, Parity bit  setup 
  U0FCR = 0x07;		/* Enable and reset TX and RX FIFO. */
  U0IER = IER_RBR  | IER_RLS;	
}

/****************************************************************************
* Name: IRQ_Timer1()
* Function: Timer1 interrupt function, inverse LED4(P0.25)
* Input(s): none
* Returns : none
****************************************************************************/
void __irq IRQ_Timer1(void) 
{  
  
  LED_Counter++;

  if( LED_Counter == 50)		// 10ms*50=0.5s led inverse
    {
      if( LED_Flag == 0) LED_Flag = 1;
      else LED_Flag = 0;
  
      LED_Counter = 0;
    }
             
   T1IR = 0x01;	    			// CLR Timer1 interrupt flag
   VICVectAddr = 0x00;			// Inform VIC interrupt is end
}


/****************************************************************************
* Name: IRQ_UART0()
* Function: UART0 interrupt
* Input(s): none
* Returns : none
****************************************************************************/
void   __irq IRQ_UART0(void)
{  
  UInt32 Temp_UART0_IRQ;
  
  if( (U0IIR & 0x0F) == 0x04 )
    {
	  Temp_UART0_IRQ = U0RBR - 48;
	  
	  switch( Temp_UART0_IRQ )
	  {
        case 1:
		  System_Status = 1;   // indicate CAN simple communication
		   break;
		case 2:
		    System_Status = 2;   // indicate CAN communication with ACC
		    break;
		case 3:
			System_Status = 3;	 // indicate CAN communication with FullCAN
		case 41:                 // receive "Y"
		case 73:				 // receive "y"
		    if( System_Status == 4) System_Status = 5;
			break;
		case 30:				 // receive "N"
		case 62:				 // receive "n"
			if( System_Status == 4 ) System_Status = 7;
			break;
	  	case 49:		         // when receive "a", reverse LED2
            Temp_UART0_IRQ = IOSET1;
            if( (Temp_UART0_IRQ & LED1CON) == 0 )
              {  
                FIO2SET = LED1CON;
              }
            else
              {  
                FIO2CLR = LED1CON;
              }
		  	break;
	  	case 50:		// when receive "b", reverse LED1	  	
		  	Temp_UART0_IRQ = IOSET1;
            if( (Temp_UART0_IRQ & LED2CON) == 0 )
              {  
                FIO2SET = LED2CON;
              }
            else
              {  
                FIO2CLR = LED2CON;
              }              
		  	break;
 		default:
			break;
      }      
    }
  
  
  Temp_UART0_IRQ = U0RBR;		//clear UART0-Rx interrupt flag, when U0IIR != 04,
  VICVectAddr = 0x00;
}


/****************************************************************************
* Name: IRQ_CAN()
* Function: IRQ_CANinterrupt
* Input(s): none
* Returns : none
****************************************************************************/
void   __irq IRQ_CAN(void)
{
   //Check the FullCAN (FCANIC0) bits 
#ifdef LPC2000_CANDRIVER_STD_FULLCAN
   if(FCANIC0 !=0 || (FCANIC1 !=0)){
   	CAN2_FULLCANRxCount++;
        CAN_FULLCAN_Rx2();	
   }
#endif
   CANStatus = CAN_RX_SR;
   if ( CANStatus & (1 << 8) )
   {
	CAN1RxCount++;
//	CAN_Rx1();
   }
   if ( CANStatus & (1 << 9) )
   {
	CAN2RxCount++;
	CAN_Rx2();
   }
   if ( CAN1GSR & (1 << 6 ) )
   {
	/* The error count includes both TX and RX */
	CAN1ErrCount = (CAN1GSR >> 16 );
   }
  if ( CAN2GSR & (1 << 6 ) )
  {
	/* The error count includes both TX and RX */
	CAN2ErrCount = (CAN2GSR >> 16 );
  }

  VICVectAddr = 0;		/* Acknowledge Interrupt */
}


/****************************************************************************
* Name: CAN_FULLCAN_Rx2()
* Function: CAN_FULLCAN_Rx2
* Input(s): none
* Returns : none
****************************************************************************/
#ifdef LPC2000_CANDRIVER_STD_FULLCAN
void CAN_FULLCAN_Rx2( void ){
plpc2000CANdriver_RXObj_t pCAN1_Rcv_Data_IRQ;
FULLCAN_MSG *FULLCAN_msgs;
unsigned int Dat1,DatA,DatB;
unsigned int msg_index = 0;
unsigned char test_bit =0;
int i =0,j=0;
unsigned int interrut_word = FCANIC0;
    for(j=0 ; j <2; j++){
	  if((j==0x0) && (FCANIC0 !=0) ){
	      interrut_word = FCANIC0;
	   }
	  else if( (j==1) && (FCANIC1 !=0) ) {
		  interrut_word = FCANIC1;
	  }
	  else{
	 	continue;
	  }
	  if(interrut_word !=0){
	   for(i=0; i<=31;i++){
	     test_bit = interrut_word	& 0x1;
	     interrut_word = interrut_word >>1;
	     if(test_bit){
	      	msg_index=i;
	      	FULLCAN_msgs= (FULLCAN_MSG *)(CAN_EOT + 0xE0038000 + msg_index*12);
	      	Dat1=FULLCAN_msgs->Dat1;
			do{
			  if((Dat1 & 0x3000000) != 0x1000000){
				  if((Dat1 & 0x3000000) == 0x3000000){
					  FULLCAN_msgs->Dat1 = Dat1 & 0xFCFFFFFF;
					  DatA = FULLCAN_msgs->DatA;
			      	  DatB = FULLCAN_msgs->DatB;
					  Dat1=FULLCAN_msgs->Dat1;
					  if((Dat1 & 0x3000000) == 0x0000){
						break;
					  }
				  }
			  }
			  Dat1=FULLCAN_msgs->Dat1;
			}while (((Dat1 & 0x3000000) != 0x0)); 
	
	   CAN2Rx_Return_Message_Flag[0] = 1;  //indicate one Rx interrupt happened
	   // put Rcv data to Rcv-Buffer, must assign value to pointer before used
	   pCAN1_Rcv_Data_IRQ = CAN_Rcv_Data + CAN_Rcv_Data_Counter;
	   pCAN1_Rcv_Data_IRQ -> FULLCALmsg = 1; 
	   pCAN1_Rcv_Data_IRQ -> RFS = Dat1;
  	   pCAN1_Rcv_Data_IRQ -> ID = Dat1 & 0x7FF;
 	   pCAN1_Rcv_Data_IRQ -> DataField[0] = DatA;
   	   pCAN1_Rcv_Data_IRQ -> DataField[1] = DatB;
	  // Return_Value = lpc2000CANdriver_ReceiveMessageCh1 (pCAN1_Rcv_Data_IRQ);
	   CAN_Rcv_Data_Counter++;
	   CAN_Rcv_Status = 1;  	   // communication successful   
	   } //if testbit
	  }	//for 0..31
	 } //if interrut_word
	}//for j 0..2
}
#endif
/******************************************************************************
** Function name:		CAN_Rx2
**
** Descriptions:		CAN Rx2 interrupt handler
**
** parameters:			None
** Returned value:		None
** 
******************************************************************************/
void CAN_Rx2( void )
{

  UInt32  Temp_CAN2Rx_IRQ=0;
  UInt8   i=0;
  plpc2000CANdriver_RXObj_t pCAN2_Rcv_Data_IRQ;


  i = 0;
  CAN2Rx_Return_Message_Flag[i] = 1;  //indicate one Rx interrupt happened
  i++;

  Temp_CAN2Rx_IRQ = CAN2ICR;
  
  if( (Temp_CAN2Rx_IRQ & 0x00000001) == 0x00000001 )
    {
	  // put Rcv data to Rcv-Buffer, must assign value to pointer before used
      pCAN2_Rcv_Data_IRQ = CAN_Rcv_Data + CAN_Rcv_Data_Counter;
      Return_Value = lpc2000CANdriver_ReceiveMessageCh2 (pCAN2_Rcv_Data_IRQ);
      CAN_Rcv_Data_Counter++;
      CAN_Rcv_Status = 1;  	   // communication successful
	}
					   
  if( ((Temp_CAN2Rx_IRQ>>2) & 0x00000001) == 0x00000001 )
    {
	  CAN2Rx_Return_Message_Flag[i] = 1;
	  //Add code for "Error Warning Interrupt-EI"	  
	}
	  i++;

  if( ((Temp_CAN2Rx_IRQ>>3) & 0x00000001) == 0x00000001 )
    {
	  CAN2Rx_Return_Message_Flag[i] = 1;
	  //Add code for "Data Overrun Interrupt-DOI"	  
	}
  i++;

  if( ((Temp_CAN2Rx_IRQ>>4) & 0x00000001) == 0x00000001 )
    {
	  CAN2Rx_Return_Message_Flag[i] = 1;
	  //Add code for "Wake-Up Interrupt-WUI"	  
	}
  i++;

  if( ((Temp_CAN2Rx_IRQ>>5) & 0x00000001) == 0x00000001 )
    {
	  CAN2Rx_Return_Message_Flag[i] = 1;
	  //Add code for "Error Passive Interrupt-EPI"	  
	}
  i++;

  if( ((Temp_CAN2Rx_IRQ>>6) & 0x00000001) == 0x00000001 )
    {
	  CAN2Rx_Return_Message_Flag[i] = 1;
	  //Add code for "Arbitration Lost Interrupt-ALI"	  
	}
  i++;

  if( ((Temp_CAN2Rx_IRQ>>7) & 0x00000001) == 0x00000001 )
    {
	  CAN2Rx_Return_Message_Flag[i] = 1;
	  //Add code for "Bus Error Interrupt-BEI"	  
	}
  i++;

  if( ((Temp_CAN2Rx_IRQ>>8) & 0x00000001) == 0x00000001 )
    {

	  //Add code for "ID Ready Interrupt-IDI"	  
	}

}



