/********************************************************************
;
;  LPC2000 - RC5 decoder
;
;  This package uses T0-CAP0 input (capture and interrupt on both edges)
;  CAP0.0 (P0.30) is connected to P0.16 (to check high / low level by software)
;
; RC5 format:
;
;      | S | F | C |   5 sytem bits    |    6 command bits     |
;      | 1 | 1 | 0 | 0 | 0 | 0 | 0 | 1 | 1 | 0 | 1 | 1 | 1 | 1 |
;
;   -----+ +-+   +-+ +-+ +-+ +-+ +---+ +-+   +---+ +-+ +-+ +-+ +----
;        | | |   | | | | | | | | |   | | |   |   | | | | | | | |
;        +-+ +---+ +-+ +-+ +-+ +-+   +-+ +---+   +-+ +-+ +-+ +-+
;
********************************************************************/

#include <LPC214x.h>                       // LPC214x definitions

#define MIN_HALF_BIT       640             // 640 us
#define HALF_BIT_TIME      889             // 889 us
#define MAX_HALF_BIT       1140            // 1140 us
#define MIN_FULL_BIT       1340            // 1340 us
#define FULL_BIT_TIME      1778            // 1778 us
#define MAX_FULL_BIT       2220            // 2220 us


unsigned char RC5_System;                  // Format 1 E/N t  s4 s3 s3 s1 s0
unsigned char RC5_Command;                 // Format 0  0  c5 c4 c3 c2 c1 c0
unsigned char RC5_flag;

static signed int    low_time;
static signed int    high_time;
static unsigned char half_bit;
static unsigned char sys;                  // system byte
static unsigned char cmd;                  // Command byte


static void RC5_Shift_Bit(char val)
{
    if (sys & 0x80)
    {
        if (cmd & 0x80)                    // command full ?
        {
            sys = 0;                       // yes, ERROR
            cmd = 0;
        }
        else
            cmd = (cmd << 1) | val;        // shift command
    }
    else
        sys = (sys << 1) | val;            // shift system
}

/************************************************************************
; RC5_Decode (we only take action at a rising edge)
;
; Half(prev) Bit   Low Time        High Time      Action     New Half Bit
; -------------------------------------------------------------------
;     0               0               0          Shift 0         0
;     0               0               1          Shift 1         1
;     0               1               0          -ERROR-         *
;     0               1               1          Shift 1,0       0
;     1               0               0          Shift 1         1
;     1               0               1          -ERROR-         *
;     1               1               0          Shift 1,0       0
;     1               1               1          -ERROR-         *
;
***********************************************************************/
static void RC5_Decode(void)
{
  unsigned char action;

    action = half_bit << 2;

// Check High Time

    if ((high_time > MIN_FULL_BIT) && (high_time < MAX_FULL_BIT))
        action = action | 1;               // high_time = long
    else if (!((high_time > MIN_HALF_BIT) && (high_time < MAX_HALF_BIT)))
    {
        sys = 0;                           // RC5 ERROR
        cmd = 0;
        return;
    }

// Check Low Time

    if ((low_time > MIN_FULL_BIT) && (low_time < MAX_FULL_BIT))
        action = action | 2;               // low_time = long
    else if (!((low_time > MIN_HALF_BIT) && (low_time < MAX_HALF_BIT)))
    {
        sys = 0;                           // RC5 ERROR
        cmd = 0;
        return;
    }

    switch (action)
    {
      case 0:  RC5_Shift_Bit(0);           // short low, short high, shift 0
               break;
      case 1:  RC5_Shift_Bit(1);           // short low, long high, shift 1
               half_bit = 1;               // new half bit is true
               break;
      case 2:  sys = 0;                    // long low, short high, ERROR
               cmd = 0;
      case 3:  RC5_Shift_Bit(1);           // long low, long high, shift 1,0
               RC5_Shift_Bit(0);
               break;
      case 4:  RC5_Shift_Bit(1);           // short low, short high, shift 1
               break;
      case 5:  sys = 0;                    // short low, long high, ERROR
               cmd = 0;
               break;
      case 6:  RC5_Shift_Bit(1);           // long low, short high, shift 1,0
               RC5_Shift_Bit(0);
               half_bit = 0;               // new half bit is false
               break;
      case 7:  sys = 0;                    // long low, long high, ERROR
               cmd = 0;
      default: break;                      // invalid
    }
}

__irq void RC5_Isr(void)
{
    T0TC = 0;                              // Reset timer

    if (T0IR & 1)                          // Timeout ?
    {                                      // to guarantee a 12 msec idle time after last RC5 pulse
        if (cmd & 0x80)                    // command full ?
        {
            RC5_Command = cmd & 0x7F;      // OK ! Save command byte
            RC5_System = sys;              // save system byte
            RC5_flag = 1;                  // set event to application
        }
        sys = 0;
        cmd = 0;
        T0IR = 0x01;                       // clear MR0 interrupt flag
    }
    else                                   // capture interrupt
    {
        if (IO0PIN & 0x00010000)           // P0.16 = P0.30 to check rising or falling edge
        {
            if (sys == 0)                  // First pulse ?
            {
                low_time  = HALF_BIT_TIME; // assume short low time
                high_time = HALF_BIT_TIME; // assume short high time
                half_bit  = 1;             // assume half bit is true
                cmd = 0x02;                // = 00000010, prepare command byte
            }
            else
                low_time = T0CR0;          // rising, so capture low time

            RC5_Decode();
        }
        else
            high_time = T0CR0;             // falling, so capture high time

        T0IR = 0x10;                       // reset interrupt flag
    }
    VICVectAddr = 0;                       // Acknowledge interrupt by reseting VIC
}

void RC5_Init(void)
{
    VICVectAddr0  = (unsigned int) &RC5_Isr;
    VICVectCntl0  = 0x24;                  // Channel0 on Source#4 ... enabled
    VICIntEnable |= 0x10;                  // Channel#4 is the Timer 0

    PINSEL1 |= 0x30000000;                 // P0.30 as CAP0.0

    T0PR  = 60;                            // prescaler 60, timer runs at 60 MHz / 60 = 1 MHz
    T0MR0 = 12000;                         // 12 ms High (idle) Timeout
    T0MCR = 3;                             // Interrupt on Match0, reset timer on match
    T0CCR = 0x0007;                        // Capture on both edges and enable the interrupt
    T0TC  = 0;                             // Reset timer
    T0TCR = 1;                             // start timer
}
