/***********************************************************************
 * $Id: fft_example.c 3473 2010-05-17 23:59:27Z nxp27266 $
 *
 * Project: M3 DSP Library Examples
 *
 * Description: Example showing how the FFT function can be used to
 *              decode DTMF tones.
 *
 * Copyright(C) 2010, NXP Semiconductor
 * All rights reserved.
 *
 ***********************************************************************
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * products. This software is supplied "AS IS" without any warranties.
 * NXP Semiconductors assumes no responsibility or liability for the
 * use of the software, conveys no license or title under any patent,
 * copyright, or mask work right to the product. NXP Semiconductors
 * reserves the right to make changes in the software without
 * notification. NXP Semiconductors also make no representation or
 * warranty that such application will be suitable for the specified
 * use without further testing or modification.
 **********************************************************************/
#include <LPC17xx.h>
#include <cr_dsplib.h>
#include <stdlib.h>
#include <math.h>

#include "RDB1768_AUDIO.h"
#include "uda1380.h"
#include "cs42l51.h"
#include "lcd.h"
#include "delay.h"
#include "whichRDB.h"
#include "dtmf.h"
#include "fft_example.h"

/* Define the gap (in pixels) between the columns used to represent each bin */
#define FFT_DISPLAY_COLUMN_GAP_PIXELS			0
/* Define the width (in pixels) of the columns used to represent each bin */
#define FFT_DISPLAY_COLUMN_WIDTH_PIXELS			2
/* Define height of area at top of display that is used for title text */
#define FFT_DISPLAY_TITLE_AREA_HEIGHT_PIXELS	32

/* Local function prototypes */
static void vDisplayTitle(void);
static void vDisplayResults(uint32_t *pu32BinData);

/* Storage for board version read at startup */
volatile uint32_t RDBversion = 0;

/* Storage for FFT input and output data, contain real and complex components */
volatile int16_t ai16FftInputData[FFT_POINTS * 2];
static int16_t ai16FftOutputData[FFT_POINTS * 2];

/* Storage for FFT magnitude results */
static uint32_t au32FftBinMagnitude[FFT_POINTS];

/* Global count of samples gathered by codec ISR */
volatile uint16_t u16SampleCount = 0;

/******************************************************************************
** Function name:   main
**
** Descriptions:    Example showing how to use the FFT function. Takes data
**                  from audio codec, performs FFT and displays result on LCD.
**
** Parameters:	    None
**
** Returned value:  None
**
******************************************************************************/
int main(void)
{
	uint8_t u8xPos = 0;

	/* General system initialization */
	SystemInit();

	/* Initialize the LCD Controller/driver */
	LCDdriver_initialisation();

	/* Display title text on LCD */
	vDisplayTitle();

	/* Check which version of RDB1768 board code is running on. */
	RDBversion = whichRDB();

	/* Enable interrupts (needed for I2C init, and audio I/O) */
	__enable_irq();

	/* Initialize the Audio Codec */
	vfAudioInit();

	while(1)
	{
		uint32_t u32MagMax = 0;

		/* Check if samples have been gathered by codec ISR */
		if (u16SampleCount == (FFT_POINTS * 2))
		{
			uint8_t u8Key;
			uint16_t u16Bin;

			/* Perform FFT on captured audio data */
			vF_dspl_fftR4b16N256((int16_t *)&ai16FftOutputData[0], (int16_t *)&ai16FftInputData[0]);

			/* Convert complex and real components into magnitude */
			for(u16Bin = 0; u16Bin < (FFT_POINTS / 2); u16Bin++)
			{
				au32FftBinMagnitude[u16Bin] = pow(ai16FftOutputData[2 * u16Bin], 2) +
						                      pow(ai16FftOutputData[(2 * u16Bin) + 1], 2);

				/* Record maximum value */
				if (au32FftBinMagnitude[u16Bin] > u32MagMax)
				{
					u32MagMax = au32FftBinMagnitude[u16Bin];
				}
			}

			/* Determine which key is pressed */
			u8Key = u8DTMF_Decode(&au32FftBinMagnitude[0], FFT_POINTS);

			/* Only display valid keys */
			if (u8Key > 0)
			{
				/* Clear key display area if all the available space has been used */
				if (u8xPos >= 128)
				{
					u8xPos = 0;
					LCD_PrintString(u8xPos, FFT_DISPLAY_TITLE_AREA_HEIGHT_PIXELS / 2, "                ", COLOR_WHITE, COLOR_BLACK);
				}

				/* Display key on LCD */
				LCD_PrintChar(u8xPos, FFT_DISPLAY_TITLE_AREA_HEIGHT_PIXELS / 2, (unsigned char)u8Key, COLOR_WHITE, COLOR_BLACK);
				u8xPos += 8;

				/* Scale magnitude values so they fit on the LCD */
				for(u16Bin = 0; u16Bin < (FFT_POINTS / 2); u16Bin++)
				{
					au32FftBinMagnitude[u16Bin] = (au32FftBinMagnitude[u16Bin] * (LCD_MAX_Y - FFT_DISPLAY_TITLE_AREA_HEIGHT_PIXELS)) / u32MagMax;
				}

				/* Display FFT results (magnitude) on LCD */
				vDisplayResults(&au32FftBinMagnitude[0]);

				/* Small delay so that sampling does not start again until current tone is complete */
				delay(150);
			}

			/* Start sampling audio data */
			u16SampleCount = 0;
		}
	}
	return 0 ;
}

/******************************************************************************
** Function name:   vDisplayTitle
**
** Descriptions:	Display title text at top of LCD.
**
** Parameters:		None
**
** Returned value:  None
**
******************************************************************************/
static void vDisplayTitle(void)
{
	LCD_PrintString(4, 0, "NXP M3 FFT Demo", COLOR_WHITE, COLOR_BLACK);
}

/******************************************************************************
** Function name:   vDisplayResults
**
** Descriptions:	Display results of FFT function as a bar graph on the LCD.
** 					Only the first half of the FFT results are displayed as the
** 					second half is simply a mirror image of the first. This
** 					function assumes that the results have been scaled so that
** 					the maximum value will fit on the screen. Note that the 0,0
**                  location is at the top right corner of the display. An area
**                  at the top of the screen is set aside for the title text.
**
** Parameters:		pu32BinData - Pointer to buffer containing the FFT results
**					to be displayed.
**
** Returned value:  None
**
******************************************************************************/
static void vDisplayResults(uint32_t *pu32BinData)
{
	uint16_t u16Bin;

	for (u16Bin = 0; u16Bin < (FFT_POINTS / 4); u16Bin++)
	{
		/* Calculate bar x-axis start position */
		uint32_t u32xStartPos = u16Bin * (FFT_DISPLAY_COLUMN_GAP_PIXELS + FFT_DISPLAY_COLUMN_WIDTH_PIXELS);
		/* Calculate bar x-axis end position */
		uint32_t u32xEndPos   = u32xStartPos + (FFT_DISPLAY_COLUMN_WIDTH_PIXELS - 1);
		/* Calculate bar y-axis start position */
		uint32_t u32yStartPos = LCD_MAX_Y - pu32BinData[u16Bin];
		/* Calculate bar y-axis end position */
		uint32_t u32yEndPos   = LCD_MAX_Y;

        /* Display bar representing magnitude of current bin */
		LCD_FilledRect(u32xStartPos, u32xEndPos, u32yStartPos, u32yEndPos, COLOR_WHITE);

		/* Fill rest of bar with background color so we don't have to clear screen each time */
		LCD_FilledRect(u32xStartPos, u32xEndPos, FFT_DISPLAY_TITLE_AREA_HEIGHT_PIXELS, u32yStartPos, COLOR_BLACK);
	}
}

/*****************************************************************************
**                            End Of File
*****************************************************************************/
