/*****************************************************************************
 *   systick.c: SYSTICK C file for NXP LPC17xx Family Microprocessors
 *
 *   Copyright(C) 2008, NXP Semiconductor
 *   All rights reserved.
 *
 *   History
 *   2008.08.21  ver 1.00    Prelimnary version, first Release
 *
******************************************************************************/
#include "LPC17xx.h"
#include "type.h"
#include "systick.h"

//extern void TCPClockHandler(void);

volatile DWORD TimeTick = 0;
volatile DWORD TimeTick2 = 0;

/* SysTick interrupt happens every 10 ms */
void SysTickHandler(void)
{
  TimeTick++;
  TimeTick2++;
  if (TimeTick >= 10) {
    TimeTick = 0;
    FIO1PIN ^= 1 << 25;
	FIO2PIN ^= 1 << 0;
  }
  if (TimeTick2 >= 20) {
    TimeTick2 = 0;
//    TCPClockHandler();  
  }  
}

void delaySysTick(DWORD tick)
{
  DWORD timetick;

  /* Enable the SysTick Counter */
  SysTick_CounterCmd(SysTick_Counter_Enable);

  timetick = TimeTick;
  while ((TimeTick - timetick) < tick);

  /* Disable SysTick Counter */
  SysTick_CounterCmd(SysTick_Counter_Disable);

  /* Clear SysTick Counter */
  SysTick_CounterCmd(SysTick_Counter_Clear);
}

/*******************************************************************************
* Function Name  : SysTick_CLKSourceConfig
* Description    : Configures the SysTick clock source.
* Input          : - SysTick_CLKSource: specifies the SysTick clock source.
*                    This parameter can be one of the following values:
*                       - SysTick_CLKSource_HCLK_Div8: AHB clock divided by 8
*                         selected as SysTick clock source.
*                       - SysTick_CLKSource_HCLK: AHB clock selected as
*                         SysTick clock source.
* Output         : None
* Return         : None
*******************************************************************************/
void SysTick_CLKSourceConfig(DWORD SysTick_CLKSource)
{
  if (SysTick_CLKSource == SysTick_CLKSource_CCLK)
  {
    NVIC_ST_CTRL |= SysTick_CLKSource_CCLK;
  }
  else
  {
    NVIC_ST_CTRL &= SysTick_CLKSource_STCLK;
  }
}

/*******************************************************************************
* Function Name  : SysTick_SetReload
* Description    : Sets SysTick Reload value.
* Input          : - Reload: SysTick Reload new value.
*                    This parameter must be a number between 1 and 0xFFFFFF.
* Output         : None
* Return         : None
*******************************************************************************/
void SysTick_SetReload(DWORD Reload)
{
  NVIC_ST_RELOAD = Reload;
}

/*******************************************************************************
* Function Name  : SysTick_CounterCmd
* Description    : Enables or disables the SysTick counter.
* Input          : - SysTick_Counter: new state of the SysTick counter.
*                    This parameter can be one of the following values:
*                       - SysTick_Counter_Disable: Disable counter
*                       - SysTick_Counter_Enable: Enable counter
*                       - SysTick_Counter_Clear: Clear counter value to 0
* Output         : None
* Return         : None
*******************************************************************************/
void SysTick_CounterCmd(DWORD SysTick_Counter)
{
  if (SysTick_Counter == SysTick_Counter_Clear)
  {
    NVIC_ST_CURRENT = SysTick_Counter_Clear;
  }
  else
  {
    if (SysTick_Counter == SysTick_Counter_Enable)
    {
      NVIC_ST_CTRL |= SysTick_Counter_Enable;
    }
    else
    {
      NVIC_ST_CTRL &= SysTick_Counter_Disable;
    }
  }
}

/*******************************************************************************
* Function Name  : SysTick_ITConfig
* Description    : Enables or disables the SysTick Interrupt.
* Input          : - NewState: new state of the SysTick Interrupt.
*                    This parameter can be: ENABLE or DISABLE.
* Output         : None
* Return         : None
*******************************************************************************/
void SysTick_ITConfig(FunctionalState NewState)
{
  if (NewState != DISABLE)
  {
    NVIC_ST_CTRL |= CTRL_TICKINT_Set;
  }
  else
  {
    NVIC_ST_CTRL &= CTRL_TICKINT_Reset;
  }
}

/*******************************************************************************
* Function Name  : SysTick_GetCounter
* Description    : Gets SysTick counter value.
* Input          : None
* Output         : None
* Return         : SysTick current value
*******************************************************************************/
DWORD SysTick_GetCounter(void)
{
  return NVIC_ST_CURRENT;
}

/*******************************************************************************
* Function Name  : SysTick_GetFlagStatus
* Description    : Checks whether the specified SysTick flag is set or not.
* Input          : - SysTick_FLAG: specifies the flag to check.
*                    This parameter can be one of the following values:
*                       - SysTick_FLAG_COUNT
*                       - SysTick_FLAG_SKEW
*                       - SysTick_FLAG_NOREF
* Output         : None
* Return         : None
*******************************************************************************/
FlagStatus SysTick_GetFlagStatus(BYTE SysTick_FLAG)
{
  DWORD tmp = 0;
  DWORD statusreg = 0;
  FlagStatus bitstatus = RESET;

  /* Get the SysTick register index */
  tmp = SysTick_FLAG >> 5;

  if (tmp == 1) /* The flag to check is in CTRL register */
  {
    statusreg = NVIC_ST_CTRL;
  }
  else          /* The flag to check is in CALIB register */
  {
    statusreg = NVIC_ST_CALIB;
  }

  /* Get the flag position */
  tmp = SysTick_FLAG & 0x1F;

  if ((statusreg & (1 << tmp)) != RESET)
  {
    bitstatus = SET;
  }
  else
  {
    bitstatus = RESET;
  }
  return bitstatus;
}

/******************************************************************************
**                            End Of File
******************************************************************************/
