/*****************************************************************************
 *   nvic.c: NVIC C file for NXP LPC17xx Family Microprocessors
 *
 *   Copyright(C) 2008, NXP Semiconductor
 *   All rights reserved.
 *
 *   History
 *   2008.08.21  ver 1.00    Prelimnary version, first Release
 *
******************************************************************************/
#include "LPC17xx.h"
#include "type.h"
#include "it.h"
#include "nvic.h"
#include "cortexm3_macro.h"

extern volatile PFV cb [MAX_IRQChannel];

/*******************************************************************************
* Function Name  : NVIC_DeInit
* Description    : Deinitializes the NVIC peripheral registers to their default
*                  reset values.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void NVIC_DeInit(void)
{
  BYTE i;

  NVIC_DISABLE0 = 0xFFFFFFFF;
  NVIC_DISABLE1 = 0x00000001;
  NVIC_UNPEND0 = 0xFFFFFFFF;
  NVIC_UNPEND1 = 0x00000001;
  
  NVIC_PRI0 = 0x00000000;
  NVIC_PRI1 = 0x00000000;
  NVIC_PRI2 = 0x00000000;
  NVIC_PRI3 = 0x00000000;
  NVIC_PRI4 = 0x00000000;
  NVIC_PRI5 = 0x00000000;
  NVIC_PRI6 = 0x00000000;
  NVIC_PRI7 = 0x00000000;
  NVIC_PRI8 = 0x00000000;

  for (i = 0; i < MAX_IRQChannel; i++)
  {
    cb[i] = NULL;
  }
}

/*******************************************************************************
* Function Name  : NVIC_SCBDeInit
* Description    : Deinitializes the SCB peripheral registers to their default 
*                  reset values.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void NVIC_SCBDeInit(void)
{
  NVIC_INT_CTRL = 0x0A000000;
  NVIC_VECT_TABLE = 0x00000000;
  NVIC_AP_INT_RST = 0x05FA0000;
  NVIC_SYS_CTRL = 0x00000000;
  NVIC_CFG_CTRL = 0x00000000;
  NVIC_SYS_H_PRI1 = 0x00000000;
  NVIC_SYS_H_PRI2 = 0x00000000;
  NVIC_SYS_H_PRI3 = 0x00000000;
  NVIC_SYS_H_CTRL = 0x00000000;
  NVIC_FAULT_STA = 0xFFFFFFFF;
  NVIC_HARD_F_STA = 0xFFFFFFFF;
  NVIC_DBG_F_STA = 0xFFFFFFFF;
}

/*******************************************************************************
* Function Name  : NVIC_PriorityGroupConfig
* Description    : Configures the priority grouping: pre-emption priority
*                  and subpriority.
* Input          : - NVIC_PriorityGroup: specifies the priority grouping bits
*                    length. This parameter can be one of the following values:
*                       - NVIC_PriorityGroup_0: 0 bits for pre-emption priority
*                         5 bits for subpriority
*                       - NVIC_PriorityGroup_1: 1 bits for pre-emption priority
*                         4 bits for subpriority
*                       - NVIC_PriorityGroup_2: 2 bits for pre-emption priority
*                         3 bits for subpriority
*                       - NVIC_PriorityGroup_3: 3 bits for pre-emption priority
*                         2 bits for subpriority
*                       - NVIC_PriorityGroup_4: 4 bits for pre-emption priority
*                         1 bits for subpriority
*                       - NVIC_PriorityGroup_5: 5 bits for pre-emption priority
*                         0 bits for subpriority
* Output         : None
* Return         : None
*******************************************************************************/
void NVIC_PriorityGroupConfig(DWORD NVIC_PriorityGroup)
{
  /* Set the PRIGROUP[10:8] bits according to NVIC_PriorityGroup value */
  NVIC_AP_INT_RST = 0x05FA0000 | NVIC_PriorityGroup;
}

/*******************************************************************************
* Function Name  : NVIC_Init
* Description    : Initializes the NVIC peripheral according to the specified
*                  parameters in the NVIC_InitStruct.
* Input          : - NVIC_InitStruct: pointer to a NVIC_InitTypeDef structure
*                    that contains the configuration information for the
*                    specified NVIC peripheral.
* Output         : None
* Return         : None
*******************************************************************************/
void NVIC_Init(NVIC_InitTypeDef* NVIC_InitStruct)
{
  DWORD tmppriority = 0x00, tmpreg = 0x00, tmpmask = 0x00;
  DWORD tmppre = 0, tmpsub = 0x1F;

  if (NVIC_InitStruct->NVIC_IRQChannelCmd != DISABLE)
  {
    /* Compute the Corresponding IRQ Priority --------------------------------*/    
    tmppriority = (0x700 - (NVIC_AP_INT_RST & 0x700))>> 0x08;
    tmppre = (0x5 - tmppriority);
    tmpsub = tmpsub >> tmppriority;
    
    tmppriority = NVIC_InitStruct->NVIC_IRQChannelPreemptionPriority << tmppre;
    tmppriority |=  NVIC_InitStruct->NVIC_IRQChannelSubPriority & tmpsub;

    tmppriority = tmppriority << 0x03;
    tmppriority = tmppriority << ((NVIC_InitStruct->NVIC_IRQChannel & 0x03) * 0x08);
    
    if ((NVIC_InitStruct->NVIC_IRQChannel >> 0x02) == 0)
	{
      tmpreg = NVIC_PRI0;
	}
	else if ((NVIC_InitStruct->NVIC_IRQChannel >> 0x02) == 1)
	{
      tmpreg = NVIC_PRI1;
	}
	else if ((NVIC_InitStruct->NVIC_IRQChannel >> 0x02) == 2)
	{
      tmpreg = NVIC_PRI2;
	}
	else if ((NVIC_InitStruct->NVIC_IRQChannel >> 0x02) == 3)
	{
      tmpreg = NVIC_PRI3;
	}
	else if ((NVIC_InitStruct->NVIC_IRQChannel >> 0x02) == 4)
	{
      tmpreg = NVIC_PRI4;
	}
	else if ((NVIC_InitStruct->NVIC_IRQChannel >> 0x02) == 5)
	{
      tmpreg = NVIC_PRI5;
	}
	else if ((NVIC_InitStruct->NVIC_IRQChannel >> 0x02) == 6)
	{
      tmpreg = NVIC_PRI6;
	}
	else if ((NVIC_InitStruct->NVIC_IRQChannel >> 0x02) == 7)
	{
      tmpreg = NVIC_PRI7;
	}
	else
	{
      tmpreg = NVIC_PRI8;
	}
    tmpmask = 0xFF << ((NVIC_InitStruct->NVIC_IRQChannel & 0x03) * 0x08);
    tmpreg &= ~tmpmask;
    tmppriority &= tmpmask;  
    tmpreg |= tmppriority;
    if ((NVIC_InitStruct->NVIC_IRQChannel >> 0x02) == 0)
	{
      NVIC_PRI0 = tmpreg;
	}
	else if ((NVIC_InitStruct->NVIC_IRQChannel >> 0x02) == 1)
	{
      NVIC_PRI1 = tmpreg;
	}
	else if ((NVIC_InitStruct->NVIC_IRQChannel >> 0x02) == 2)
	{
      NVIC_PRI2 = tmpreg;
	}
	else if ((NVIC_InitStruct->NVIC_IRQChannel >> 0x02) == 3)
	{
      NVIC_PRI3 = tmpreg;
	}
	else if ((NVIC_InitStruct->NVIC_IRQChannel >> 0x02) == 4)
	{
      NVIC_PRI4 = tmpreg;
	}
	else if ((NVIC_InitStruct->NVIC_IRQChannel >> 0x02) == 5)
	{
      NVIC_PRI5 = tmpreg;
	}
	else if ((NVIC_InitStruct->NVIC_IRQChannel >> 0x02) == 6)
	{
      NVIC_PRI6 = tmpreg;
	}
	else if ((NVIC_InitStruct->NVIC_IRQChannel >> 0x02) == 7)
	{
      NVIC_PRI7 = tmpreg;
	}
	else
	{
      NVIC_PRI8 = tmpreg;
	}

    /* Enable the Selected IRQ Channels --------------------------------------*/
    if (NVIC_InitStruct->NVIC_IRQChannel >> 0x05)
	{
      NVIC_ENABLE1 = 0x01 << (NVIC_InitStruct->NVIC_IRQChannel & 0x1F);
    }
	else
	{
      NVIC_ENABLE0 = 0x01 << (NVIC_InitStruct->NVIC_IRQChannel & 0x1F);
	}

	cb[NVIC_InitStruct->NVIC_IRQChannel] = NVIC_InitStruct->cb;
  }
  else
  {
    /* Disable the Selected IRQ Channels -------------------------------------*/
    if (NVIC_InitStruct->NVIC_IRQChannel >> 0x05)
	{
      NVIC_DISABLE1 = 0x01 << (NVIC_InitStruct->NVIC_IRQChannel & 0x1F);
    }
	else
	{
      NVIC_DISABLE0 = 0x01 << (NVIC_InitStruct->NVIC_IRQChannel & 0x1F);
	}
  }
}

/*******************************************************************************
* Function Name  : NVIC_StructInit
* Description    : Fills each NVIC_InitStruct member with its default value.
* Input          : - NVIC_InitStruct: pointer to a NVIC_InitTypeDef structure which
*                    will be initialized.
* Output         : None
* Return         : None
*******************************************************************************/
void NVIC_StructInit(NVIC_InitTypeDef* NVIC_InitStruct)
{
  /* NVIC_InitStruct members default value */
  NVIC_InitStruct->NVIC_IRQChannel = 0x00;
  NVIC_InitStruct->NVIC_IRQChannelPreemptionPriority = 0x00;
  NVIC_InitStruct->NVIC_IRQChannelSubPriority = 0x00;
  NVIC_InitStruct->NVIC_IRQChannelCmd = DISABLE;
}

/*******************************************************************************
* Function Name  : NVIC_SETPRIMASK
* Description    : Enables the PRIMASK priority: Raises the execution priority to 0.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void NVIC_SETPRIMASK(void)
{
  __SETPRIMASK();
}

/*******************************************************************************
* Function Name  : NVIC_RESETPRIMASK
* Description    : Disables the PRIMASK priority.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void NVIC_RESETPRIMASK(void)
{
  __RESETPRIMASK();
}

/*******************************************************************************
* Function Name  : NVIC_SETFAULTMASK
* Description    : Enables the FAULTMASK priority: Raises the execution priority to -1.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void NVIC_SETFAULTMASK(void)
{
  __SETFAULTMASK();
}

/*******************************************************************************
* Function Name  : NVIC_RESETFAULTMASK
* Description    : Disables the FAULTMASK priority.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void NVIC_RESETFAULTMASK(void)
{
  __RESETFAULTMASK();
}

/*******************************************************************************
* Function Name  : NVIC_BASEPRICONFIG
* Description    : The execution priority can be changed from 15 (lowest 
                   configurable priority) to 1.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void NVIC_BASEPRICONFIG(DWORD NewPriority)
{
  __BASEPRICONFIG(NewPriority << 0x03);
}

/*******************************************************************************
* Function Name  : NVIC_GetBASEPRI
* Description    : Returns the BASEPRI mask value.
* Input          : None
* Output         : None
* Return         : BASEPRI register value
*******************************************************************************/
DWORD NVIC_GetBASEPRI(void)
{
  return __GetBASEPRI();
}

/*******************************************************************************
* Function Name  : NVIC_GetCurrentPendingIRQChannel
* Description    : Returns the current pending IRQ channel identifier.
* Input          : None
* Output         : None
* Return         : Pending IRQ Channel Identifier.
*******************************************************************************/
WORD NVIC_GetCurrentPendingIRQChannel(void)
{
  return (WORD)((NVIC_INT_CTRL & 0x003FF000) >> 0x0C);
}

/*******************************************************************************
* Function Name  : NVIC_GetIRQChannelPendingBitStatus
* Description    : Checks whether the specified IRQ Channel pending bit is set
*                  or not.
* Input          : - NVIC_IRQChannel: specifies the interrupt pending bit to check.
* Output         : None
* Return         : The new state of IRQ Channel pending bit(SET or RESET).
*******************************************************************************/
ITStatus NVIC_GetIRQChannelPendingBitStatus(BYTE NVIC_IRQChannel)
{
  ITStatus pendingirqstatus = RESET;
  DWORD tmp = 0x00;
  
  tmp = (0x01 << (NVIC_IRQChannel & 0x1F));

  if (NVIC_IRQChannel >> 0x05)
  {
    if ((NVIC_PEND1 & tmp) == tmp)
    {
      pendingirqstatus = SET;
    }
    else
    {
      pendingirqstatus = RESET;
    }
  }
  else
  {
    if ((NVIC_PEND0 & tmp) == tmp)
    {
      pendingirqstatus = SET;
    }
    else
    {
      pendingirqstatus = RESET;
    }
  }
  return pendingirqstatus;
}

/*******************************************************************************
* Function Name  : NVIC_SetIRQChannelPendingBit
* Description    : Sets the NVICs interrupt pending bit.
* Input          : - NVIC_IRQChannel: specifies the interrupt pending bit to Set.
* Output         : None
* Return         : None
*******************************************************************************/
void NVIC_SetIRQChannelPendingBit(BYTE NVIC_IRQChannel)
{
  NVIC_SW_TRIG = (DWORD)NVIC_IRQChannel;
}

/*******************************************************************************
* Function Name  : NVIC_ClearIRQChannelPendingBit
* Description    : Clears the NVICs interrupt pending bit.
* Input          : - NVIC_IRQChannel: specifies the interrupt pending bit to clear.
* Output         : None
* Return         : None
*******************************************************************************/
void NVIC_ClearIRQChannelPendingBit(BYTE NVIC_IRQChannel)
{
  if (NVIC_IRQChannel >> 0x05)
  {
    NVIC_UNPEND1 = 0x01 << (NVIC_IRQChannel & 0x1F);
  }
  else
  {
    NVIC_UNPEND0 = 0x01 << (NVIC_IRQChannel & 0x1F);
  }
}

/*******************************************************************************
* Function Name  : NVIC_GetCurrentActiveHandler
* Description    : Returns the current active Handler (IRQ Channel and
*                  SystemHandler) identifier.
* Input          : None
* Output         : None
* Return         : Active Handler Identifier.
*******************************************************************************/
WORD NVIC_GetCurrentActiveHandler(void)
{
  return (WORD)(NVIC_INT_CTRL & 0x1FF);
}

/*******************************************************************************
* Function Name  : NVIC_GetIRQChannelActiveBitStatus
* Description    : Checks whether the specified IRQ Channel active bit is set
*                  or not.
* Input          : - NVIC_IRQChannel: specifies the interrupt active bit to check.
* Output         : None
* Return         : The new state of IRQ Channel active bit(SET or RESET).
*******************************************************************************/
ITStatus NVIC_GetIRQChannelActiveBitStatus(BYTE NVIC_IRQChannel)
{
  ITStatus activeirqstatus = RESET;
  DWORD tmp = 0x00;

  tmp = (0x01 << (NVIC_IRQChannel & 0x1F));

  if (NVIC_IRQChannel >> 0x05)
  {
    if ((NVIC_ACTIVE1 & tmp) == tmp )
    {
      activeirqstatus = SET;
    }
    else
    {
      activeirqstatus = RESET;
    }
  }
  else
  {
    if ((NVIC_ACTIVE0 & tmp) == tmp )
    {
      activeirqstatus = SET;
    }
    else
    {
      activeirqstatus = RESET;
    }
  }
  return activeirqstatus;
}

/*******************************************************************************
* Function Name  : NVIC_GetCPUID
* Description    : Returns the ID number, the version number and the implementation
*                  details of the Cortex-M3 core.
* Input          : None
* Output         : None
* Return         : CPU ID.
*******************************************************************************/
DWORD NVIC_GetCPUID(void)
{
  return NVIC_CPUID;
}

/*******************************************************************************
* Function Name  : NVIC_SetVectorTable
* Description    : Sets the vector table location and Offset.
* Input          : - NVIC_VectTab: specifies if the vector table is in RAM or
*                    FLASH memory.
*                    This parameter can be one of the following values:
*                       - NVIC_VectTab_RAM
*                       - NVIC_VectTab_FLASH
*                  - Offset: Vector Table base offset field. 
*                            This value must be a multiple of 0x100.
* Output         : None
* Return         : None
*******************************************************************************/
void NVIC_SetVectorTable(DWORD NVIC_VectTab, DWORD Offset)
{ 
  NVIC_VECT_TABLE = NVIC_VectTab | (Offset & 0x1FFFFF80);
}

/*******************************************************************************
* Function Name  : NVIC_GenerateSystemReset
* Description    : Generates a system reset.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void NVIC_GenerateSystemReset(void)
{
  NVIC_AP_INT_RST = 0x05FA0000 | 0x04;
}

/*******************************************************************************
* Function Name  : NVIC_GenerateCoreReset
* Description    : Generates a Core (Core + NVIC) reset.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void NVIC_GenerateCoreReset(void)
{
  NVIC_AP_INT_RST = 0x05FA0000 | 0x01;
}

/*******************************************************************************
* Function Name  : NVIC_SystemLPConfig
* Description    : Selects the condition for the system to enter low power mode.
* Input          : - LowPowerMode: Specifies the new mode for the system to enter
*                    low power mode.
*                    This parameter can be one of the following values:
*                       - NVIC_LP_SEVONPEND
*                       - NVIC_LP_SLEEPDEEP
*                       - NVIC_LP_SLEEPONEXIT
*                  - NewState: new state of LP condition.
*                    This parameter can be: ENABLE or DISABLE.
* Output         : None
* Return         : None
*******************************************************************************/
void NVIC_SystemLPConfig(BYTE LowPowerMode, FunctionalState NewState)
{
  if (NewState != DISABLE)
  {
    NVIC_SYS_CTRL |= (DWORD)LowPowerMode;
  }
  else
  {
    NVIC_SYS_CTRL &= ~(DWORD)LowPowerMode;
  }
}

/*******************************************************************************
* Function Name  : NVIC_SystemHandlerConfig
* Description    : Enables or disables the specified System Handlers.
* Input          : - SystemHandler: specifies the system handler to be enabled
*                    or disabled.
*                    This parameter can be one of the following values:
*                       - SystemHandler_MemoryManage
*                       - SystemHandler_BusFault
*                       - SystemHandler_UsageFault
*                  - NewState: new state of  specified System Handlers.
*                    This parameter can be: ENABLE or DISABLE.
* Output         : None
* Return         : None
*******************************************************************************/
void NVIC_SystemHandlerConfig(DWORD SystemHandler, FunctionalState NewState)
{
  DWORD tmpreg = 0x00;

  tmpreg =  0x01 << (SystemHandler & 0x1F);

  if (NewState != DISABLE)
  {
    NVIC_SYS_H_CTRL |= tmpreg;
  }
  else
  {
    NVIC_SYS_H_CTRL &= ~tmpreg;
  }
}

/*******************************************************************************
* Function Name  : NVIC_SystemHandlerPriorityConfig
* Description    : Configures the specified System Handlers priority.
* Input          : - SystemHandler: specifies the system handler to be
*                    enabled or disabled.
*                    This parameter can be one of the following values:
*                       - SystemHandler_MemoryManage
*                       - SystemHandler_BusFault
*                       - SystemHandler_UsageFault
*                       - SystemHandler_SVCall
*                       - SystemHandler_DebugMonitor
*                       - SystemHandler_PSV
*                       - SystemHandler_SysTick
*                  - SystemHandlerPreemptionPriority: new priority group of the
*                    specified system handlers.
*                  - SystemHandlerSubPriority: new sub priority of the specified
*                    system handlers.
* Output         : None
* Return         : None
*******************************************************************************/
void NVIC_SystemHandlerPriorityConfig(DWORD SystemHandler, BYTE SystemHandlerPreemptionPriority,
                                      BYTE SystemHandlerSubPriority)
{
  DWORD tmp1 = 0x00, tmp2 = 0x1F, handlermask = 0x00;
  DWORD tmppriority = 0x00;

  tmppriority = (0x700 - (NVIC_AP_INT_RST & 0x700))>> 0x08;
  tmp1 = (0x5 - tmppriority);
  tmp2 = tmp2 >> tmppriority;
    
  tmppriority = SystemHandlerPreemptionPriority << tmp1;
  tmppriority |=  SystemHandlerSubPriority & tmp2;
  tmppriority = tmppriority << 0x03;

  tmp1 = SystemHandler & 0xC0;
  tmp1 = tmp1 >> 0x06; 

  tmp2 = (SystemHandler >> 0x08) & 0x03;
  tmppriority = tmppriority << (tmp2 * 0x08);
  handlermask = 0xFF << (tmp2 * 0x08);
  
  if (tmp1 == 0x00)
  {
    NVIC_SYS_H_PRI1 &= ~handlermask;
    NVIC_SYS_H_PRI1 |= tmppriority;
  }
  else if (tmp1 == 0x01)
  {
    NVIC_SYS_H_PRI2 &= ~handlermask;
    NVIC_SYS_H_PRI2 |= tmppriority;
  }
  else if (tmp1 == 0x02)
  {
    NVIC_SYS_H_PRI3 &= ~handlermask;
    NVIC_SYS_H_PRI3 |= tmppriority;
  }
}

/*******************************************************************************
* Function Name  : NVIC_GetSystemHandlerPendingBitStatus
* Description    : Checks whether the specified System handlers pending bit is
*                  set or not.
* Input          : - SystemHandler: specifies the system handler pending bit to
*                    check.
*                    This parameter can be one of the following values:
*                       - SystemHandler_MemoryManage
*                       - SystemHandler_BusFault
*                       - SystemHandler_SVCall
* Output         : None
* Return         : The new state of System Handler pending bit(SET or RESET).
*******************************************************************************/
ITStatus NVIC_GetSystemHandlerPendingBitStatus(DWORD SystemHandler)
{
  ITStatus bitstatus  = RESET;
  DWORD tmp = 0x00, tmppos = 0x00;

  tmppos = (SystemHandler >> 0x0A);
  tmppos &= 0x0F;

  tmppos = 0x01 << tmppos;

  tmp = NVIC_SYS_H_CTRL & tmppos;

  if (tmp == tmppos)
  {
    bitstatus = SET;
  }
  else
  {
    bitstatus = RESET;
  }
  return bitstatus;
}

/*******************************************************************************
* Function Name  : NVIC_SetSystemHandlerPendingBit
* Description    : Sets System Handler pending bit.
* Input          : - SystemHandler: specifies the system handler pending bit
*                    to be set.
*                    This parameter can be one of the following values:
*                       - SystemHandler_NMI
*                       - SystemHandler_PSV
*                       - SystemHandler_SysTick
* Output         : None
* Return         : None
*******************************************************************************/
void NVIC_SetSystemHandlerPendingBit(DWORD SystemHandler)
{
  DWORD tmp = 0x00;

  /* Get the System Handler pending bit position */
  tmp = SystemHandler & 0x1F;
  /* Set the corresponding System Handler pending bit */
  NVIC_INT_CTRL |= (0x01 << tmp);
}

/*******************************************************************************
* Function Name  : NVIC_ClearSystemHandlerPendingBit
* Description    : Clears System Handler pending bit.
* Input          : - SystemHandler: specifies the system handler pending bit to
*                    be clear.
*                    This parameter can be one of the following values:
*                       - SystemHandler_PSV
*                       - SystemHandler_SysTick
* Output         : None
* Return         : None
*******************************************************************************/
void NVIC_ClearSystemHandlerPendingBit(DWORD SystemHandler)
{
  DWORD tmp = 0x00;

  /* Get the System Handler pending bit position */
  tmp = SystemHandler & 0x1F;
  /* Clear the corresponding System Handler pending bit */
  NVIC_INT_CTRL |= (0x01 << (tmp - 0x01));
}

/*******************************************************************************
* Function Name  : NVIC_GetSystemHandlerActiveBitStatus
* Description    : Checks whether the specified System handlers active bit is
*                  set or not.
* Input          : - SystemHandler: specifies the system handler active bit to
*                    check.
*                    This parameter can be one of the following values:
*                       - SystemHandler_MemoryManage
*                       - SystemHandler_BusFault
*                       - SystemHandler_UsageFault
*                       - SystemHandler_SVCall
*                       - SystemHandler_DebugMonitor
*                       - SystemHandler_PSV
*                       - SystemHandler_SysTick
* Output         : None
* Return         : The new state of System Handler active bit(SET or RESET).
*******************************************************************************/
ITStatus NVIC_GetSystemHandlerActiveBitStatus(DWORD SystemHandler)
{
  ITStatus bitstatus  = RESET;

  DWORD tmp = 0x00, tmppos = 0x00;

  tmppos = (SystemHandler >> 0x0E) & 0x0F;

  tmppos = 0x01 << tmppos;

  tmp = NVIC_SYS_H_CTRL & tmppos;

  if (tmp == tmppos)
  {
    bitstatus = SET;
  }
  else
  {
    bitstatus = RESET;
  }
  return bitstatus;
}

/*******************************************************************************
* Function Name  : NVIC_GetFaultHandlerSources
* Description    : Returns the system fault handlers sources.
* Input          : - SystemHandler: specifies the system handler to get its fault
*                    sources.
*                    This parameter can be one of the following values:
*                       - SystemHandler_HardFault
*                       - SystemHandler_MemoryManage
*                       - SystemHandler_BusFault
*                       - SystemHandler_UsageFault
*                       - SystemHandler_DebugMonitor
* Output         : None
* Return         : Source of the fault handler.
*******************************************************************************/
DWORD NVIC_GetFaultHandlerSources(DWORD SystemHandler)
{
  DWORD faultsources = 0x00;
  DWORD tmpreg = 0x00, tmppos = 0x00;

  tmpreg = (SystemHandler >> 0x12) & 0x03;
  tmppos = (SystemHandler >> 0x14) & 0x03;

  if (tmpreg == 0x00)
  {
    faultsources = NVIC_HARD_F_STA;
  }
  else if (tmpreg == 0x01)
  {
    faultsources = NVIC_FAULT_STA >> (tmppos * 0x08);
    if (tmppos != 0x02)
    {
      faultsources &= 0xFF;
    }
    else
    {
      faultsources &= 0xFFFF;
    }
  }
  else
  {
    faultsources = NVIC_DBG_F_STA;
  }
  return faultsources;
}

/*******************************************************************************
* Function Name  : NVIC_GetFaultAddress
* Description    : Returns the address of the location that generated a fault
*                  handler.
* Input          : - SystemHandler: specifies the system handler to get its
*                    fault address.
*                    This parameter can be one of the following values:
*                       - SystemHandler_MemoryManage
*                       - SystemHandler_BusFault
* Output         : None
* Return         : Fault address.
*******************************************************************************/
DWORD NVIC_GetFaultAddress(DWORD SystemHandler)
{
  DWORD faultaddress = 0x00;
  DWORD tmp = 0x00;

  tmp = (SystemHandler >> 0x16) & 0x01;

  if (tmp == 0x00)
  {
    faultaddress = NVIC_MM_F_ADR;
  }
  else
  {
    faultaddress = NVIC_BUS_F_ADR;
  }
  return faultaddress;
}

/******************************************************************************
**                            End Of File
******************************************************************************/
