/******************************************************************************/
/* RTC Calibration Example													  */
/******************************************************************************/
/* This file is part of the uVision/ARM development tools.                    */
/* Copyright (c) 2005-2007 Keil Software. All rights reserved.                */
/* This software may only be used under the terms of a valid, current,        */
/* end user licence from KEIL for a compatible version of KEIL software       */
/* development tools. Nothing else gives you the right to use this software.  */
/******************************************************************************/

#include "LPC17xx.h"
#include "rtc.h"
#include "timer.h"
#include "GLCD.h"
#include <stdio.h>

#define TIMER_0_LED_VALUE_OUTPUT	1

/*****************************************************************************
** Function name:		init_GPIO
**
** Descriptions:		Sets up the 8 LEDs of Keil's 17xx board as outputs.
**
** parameters:			None			 
** 						
** Returned value:		None
** 
*****************************************************************************/
void init_GPIO(void) {
	/* Set GPIO LEDs P2.2 - P2.6 as output */
	LPC_GPIO1->FIODIR	|=	((1<<28)| (1<<29)| (1<<31));
	/* Set GPIO LEDs P1.28, P1.29, P1.31 as output */
	LPC_GPIO2->FIODIR	|=  ((1<<2) | (1<<3) | (1<<4) | (1<<5) | (1<<6));
}

/*****************************************************************************
** Function name:		output_4_bit_value
**
** Descriptions:		Function to output a 4-bit value onto the upper or
**						lower LEDs of Keil's 17xx board.
**
** parameters:			value: value to be displayed (only the lower 
**								4 bits are used)			 
**						high_or_low_byte:	'1', output onto upper 4 LEDs
**											'0', output onto lower 4 LEDs
** 						
** Returned value:		None
** 
*****************************************************************************/
void output_4_bit_value(uint32_t value, uint8_t high_or_low_byte) {

	/* Only save the lower 4 bits */	
	uint8_t		temp_byte =  (value & 0x0F);
	uint32_t	GPIO1_mask, GPIO2_mask;
	
	if(high_or_low_byte){ 	/* Upper nibble */
		/* Save current GPIO masks */
		GPIO1_mask = LPC_GPIO1->FIOMASK;
		GPIO2_mask = LPC_GPIO2->FIOMASK;
		/* Set GPIO masks for the GPIO */
		LPC_GPIO1->FIOMASK = ~((1<<28)| (1<<29)| (1<<31) );
		LPC_GPIO2->FIOMASK = ~(1<<2);
		
		/* Set new 4bit value onto the upper 4 LEDs */
		LPC_GPIO1->FIOPIN	= 	 (	((temp_byte & 0x8) << 25)	|
								((temp_byte & 0x4) << 27)	|
								((temp_byte & 0x2) << 30)	);

		LPC_GPIO2->FIOPIN	=	 (  ((temp_byte & 0x1) << 2 )   );

		/* Restore Masks */
		LPC_GPIO1->FIOMASK = GPIO1_mask;
		LPC_GPIO2->FIOMASK = GPIO2_mask;
	}else {					/* Lower nibble */
		/* Save current GPIO masks */
		GPIO2_mask = LPC_GPIO2->FIOMASK; 	
	
		/* Set GPIO masks for the GPIO */
		LPC_GPIO2->FIOMASK = ~((1<<3) | (1<<4) | (1<<5) | (1<<6));

		/* Set new 4bit value onto the lower 4 LEDs */
		LPC_GPIO2->FIOPIN	=	(	((temp_byte & 0x8)	   )	|
								((temp_byte & 0x4) << 2)	|
								((temp_byte & 0x2) << 4)	|
								((temp_byte & 0x1) << 6)	);

		/* Restore Masks */
		LPC_GPIO2->FIOMASK = GPIO2_mask;
	}
	return;
}

int main () {
	unsigned char *text[5] = {"NXP Semiconductors",
							  "RTC Calibration DEMO",
							  "Unmodified Timer 0",
							  "Corrected  RTC",
							  "      DD:HH:MM:SS"};

   	unsigned char temp[21];
	int calibration_value;

	/* Initialize LED GPIO Pins */
	init_GPIO();
	SystemInit();

	/* Initialize GLCD */
	GLCD_init();
	GLCD_clear(White);
	GLCD_setTextColor(Black);
	GLCD_setBackColor(White);

	GLCD_displayStringLn(Line0, text[0]);
	GLCD_displayStringLn(Line1, text[1]);
	
	GLCD_displayStringLn(Line4, text[2]);
	GLCD_displayStringLn(Line6, text[3]);

	/* Initialize Timer0 */
	init_timer(0, 60*60*24*3);	//86400 seconds in 3 days
	disable_timer(0);
	reset_timer(0);

	/* Initialize RTC with RTCCAL and RTCDIR values */
	InitRTC();

	/* Start Timer 0 */
	enable_timer(0);
	/* start RTC */
	StartRTC();

	/* Output the calibration direction */
	/* Convert integer values to char array format */
	if      ( ((LPC_RTC->CALIBRATION) & (1<<17)) && (!((LPC_RTC->CCR)& (1<<4)))){
		/* Backward calibration and autocalibration enabled */
		sprintf((char *)temp, "%s", "Backward adjustment");
	}else if(!((LPC_RTC->CALIBRATION) & (1<<17)) && (!((LPC_RTC->CCR)& (1<<4)))){
		/* Forward calibration and auto calibration enabled */
		sprintf((char *)temp, "%s", "Forward  adjustment");
	}else{
		/* Auto calibration is NOT enabled */
		sprintf((char *)temp, "%s", "No adjustment needed");
	}
	GLCD_displayStringLn(Line8, temp);
	
	calibration_value = (LPC_RTC->CALIBRATION) & 0x1FFFF;
	/* Output the time length until adjustment */
	sprintf((char *)temp, "every %2.2d:%2.2d:%2.2d:%2.2d", 
										   (calibration_value/86400)%356,
										   (calibration_value/3600)%24,
										   (calibration_value%3600)/60,
										   (calibration_value%3600)%60);
	GLCD_displayStringLn(Line9, temp);

	GLCD_setTextColor(Red);
	GLCD_displayStringLn(Line3, text[4]);
	while(1){
		#if TIMER_0_LED_VALUE_OUTPUT
			/* Display # of seconds from Timer0 to the upper nibble */
			output_4_bit_value(LPC_TIM0->TC % 60 , 1);
		#else
			/* Toggle P1.28 each second */
			output_4_bit_value(LPC_TIM0->TC << 3, 1);
		#endif
			/* Display RTC's SECONDS register onto the lower nibble */
			output_4_bit_value(LPC_RTC->SEC , 0);
			
			/* Convert integer values to char array format and output onto the Graphics LCD */
				/* Timer 0 */
			sprintf((char *)temp,"TIM0: %2.2d:%2.2d:%2.2d:%2.2d",  (LPC_TIM0->TC/86400)%356, 
																   (LPC_TIM0->TC/3600)%24,
																  ((LPC_TIM0->TC)%3600)/60,
																  ((LPC_TIM0->TC)%3600)%60);
			GLCD_displayStringLn(Line5, temp);
				
				/* RTC */
			sprintf((char *)temp,"RTC : %2.2d:%2.2d:%2.2d:%2.2d",	LPC_RTC->DOY,
																	LPC_RTC->HOUR,
																	LPC_RTC->MIN,
																	LPC_RTC->SEC);
			GLCD_displayStringLn(Line7, temp);			
	
	}	/*  END WHILE(1)  */
}		/*  END MAIN  */
