/*****************************************************************************
 *   i2ctest.c:  main C entry file for NXP LPC13xx Family Microprocessors
 *
 *   Copyright(C) 2009, NXP Semiconductor
 *   All rights reserved.
 *
 *   History
 *   2009.10.22  ver 1.00    Prelimnary version, first Release
 *
******************************************************************************/
#include <LPC13xx.h>			/* LPC13xx Peripheral Registers */
#include "type.h"
#include "i2c.h"

extern volatile uint32_t I2CCount;
extern volatile uint8_t I2CMasterBuffer[BUFSIZE];
extern volatile uint8_t I2CSlaveBuffer[BUFSIZE];
extern volatile uint32_t I2CMasterState;
extern volatile uint32_t I2CReadLength, I2CWriteLength;

/*******************************************************************************
**   Main Function  main()
*******************************************************************************/
int main (void)
{
  uint32_t i;
  uint8_t temp = 0, temp1 = 0;


  SystemInit();

#if __DEBUG_RAM
  SCB->VTOR = NVIC_VectTab_RAM; 
#else
  SCB->VTOR = NVIC_VectTab_FLASH; 
#endif
 
  if ( I2CInit( (uint32_t)I2CMASTER ) == FALSE )	/* initialize I2c */
  {
	while ( 1 );				/* Fatal error */
  }


  /* In order to start the I2CEngine, the all the parameters 
  must be set in advance, including I2CWriteLength, I2CReadLength,
  I2CCmd, and the I2cMasterBuffer which contains the stream
  command/data to the I2c slave device. 
  (1) If it's a I2C write only, the number of bytes to be written is 
  I2CWriteLength, I2CReadLength is zero, the content will be filled 
  in the I2CMasterBuffer. 
  (2) If it's a I2C read only, the number of bytes to be read is 
  I2CReadLength, I2CWriteLength is 0, the read value will be filled 
  in the I2CMasterBuffer. 
  (3) If it's a I2C Write/Read with repeated start, specify the 
  I2CWriteLength, fill the content of bytes to be written in 
  I2CMasterBuffer, specify the I2CReadLength, after the repeated 
  start and the device address with RD bit set, the content of the 
  reading will be filled in I2CMasterBuffer index at 
  I2CMasterBuffer[I2CWriteLength+2]. 
  
  e.g. Start, DevAddr(W), WRByte1...WRByteN, Repeated-Start, DevAddr(R), 
  RDByte1...RDByteN Stop. The content of the reading will be filled 
  after (I2CWriteLength + two devaddr) bytes. */


  /* PCA9632 initialization */
  /* Write SLA(W), CON, 00H ~ 0Ch */
  I2CWriteLength = 15;
  I2CReadLength = 0;
  I2CMasterBuffer[0] = PCA9632_ADDR;
  I2CMasterBuffer[1] = 0x80;//10100000b;		// Control register
  I2CMasterBuffer[2] = 0x81;//10100001b;		// MODE1
  I2CMasterBuffer[3] = 0x01;//00000001b;		// MODE2
  I2CMasterBuffer[4] = 0x0F;//11111111b;		// PWM0
  I2CMasterBuffer[5] = 0x1F;//11111111b;		// PWM1
  I2CMasterBuffer[6] = 0x2F;//11111111b;		// PWM2
  I2CMasterBuffer[7] = 0x0f;//11111111b;		// PWM3
  I2CMasterBuffer[8] = 0x3F;//11110000b;		// GRPPWM
  I2CMasterBuffer[9] = 0x1F;//00000000b;		// GRPFREQ
  I2CMasterBuffer[10] = 0xFF;//11111111b;		// LEDOUT
  I2CMasterBuffer[11] = 0xE2;//11100010b;		// SUBADR1
  I2CMasterBuffer[12] = 0xE4;//11100100b;		// SUBADR2
  I2CMasterBuffer[13] = 0xE8;//11101000b;		// SUBADR3
  I2CMasterBuffer[14] = 0xE0;//11100000b;		// ALLCALLADR
  I2CEngine();

while(1)
{
  for ( i = 0; i < 2000; i++ )
  {
	I2CSlaveBuffer[i] = 0x00;
  }

  /* Write PCA9674 0xFF to config IOs as input and read IOs back */  
  /* Write SLA(W), data(0xFF), SLA(R), and read 2 byte back. */
  I2CWriteLength = 2;
  I2CReadLength = 2;
  I2CMasterBuffer[0] = PCA9674_ADDR;
  I2CMasterBuffer[1] = 0xFF;
  I2CMasterBuffer[2] = PCA9674_ADDR | RD_BIT;
  I2CEngine();

  /* PCA9674 P0 as switch for FM+(1Mbps) and standard/FM(400Kbps) I2C */
  if (I2CSlaveBuffer[0] & 0x01)
  {
  	/* I2C FM+ mode */
	LPC_IOCON->PIO0_4      &= ~0x0000031F;		/*  I2C FM+ I/O config */
	LPC_IOCON->PIO0_4      |= 0x00000101;		/* I2C FM+ SCL */
	LPC_IOCON->PIO0_5      &= ~0x0000031F;	
	LPC_IOCON->PIO0_5      |= 0x00000101;		/* I2C FM+ SDA */
	/*--- Reset Baud Rate to 1Mbps ---*/
	LPC_I2C->SCLL   = 48;//24;//0x18;
	LPC_I2C->SCLH   = 48;//24;//0x18;
  }
  else
  {
  	/* I2C standard/FM */
	LPC_IOCON->PIO0_4      &= ~0x1F;		/*  I2C I/O config */
	LPC_IOCON->PIO0_4      |= 0x01;		/* I2C SCL */
	LPC_IOCON->PIO0_5      &= ~0x1F;	
	LPC_IOCON->PIO0_5      |= 0x01;		/* I2C SDA */
	/*--- Reset Baud Rate to 62.5Kbps ---*/
	LPC_I2C->SCLL   = 96;//192;//384;//2400;//0x180;
	LPC_I2C->SCLH   = 96;//192;//384;//2400;//0x180;
  }

  /* Write SLA(W), data1 */
  I2CWriteLength = 2;
  I2CReadLength = 0;
  I2CMasterBuffer[0] = PCA9674_ADDR;
  I2CMasterBuffer[1] = (I2CSlaveBuffer[0] << 4) | 0x0F;
  I2CEngine();


  /* Write PCA9632 config parameters for LED dimming */  
  /* Write SLA(W), CON, 00H ~ 0Ch */
  I2CWriteLength = 6;
  I2CReadLength = 0;
  I2CMasterBuffer[0] = PCA9632_ADDR;
  I2CMasterBuffer[1] = 0xC2;			// Control register
  I2CMasterBuffer[2] = temp;			// PWM0
  I2CMasterBuffer[3] = temp + 0x40;		// PWM1
  I2CMasterBuffer[4] = temp + 0x80;		// PWM2
  I2CMasterBuffer[5] = temp + 0xC0;		// PWM3

  I2CEngine();

  /* update the brightness */
  temp1++;
  temp = temp + (((temp1 % 2) == 0) ? 1 : 0);
}
  return 0;
}

/******************************************************************************
**                            End Of File
******************************************************************************/
